//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQml 2.12
import QtQuick 2.12

import wesual.Ui 1.0

import wesual.Ui.Private 1.0

FocusScope {
    id : uiDateEdit

    signal editingFinished()

    property date date
    property date minimumDate
    property date maximumDate

    property alias font  : edit.font
    property alias color : edit.color

    property bool  bindable : false
    property bool  readOnly : false
    property bool  allowInvalid : false

    readonly property bool hover : edit.hovered

    readonly property var locale : Qt.locale()
    readonly property string dateFormat : {
        let format = locale.dateFormat(Locale.ShortFormat);
        if (format.indexOf('yyyy') < 0) {
            format = format.replace('yy', 'yyyy');
        }

        return format;
    }
    readonly property var parseFormats : {
        const formats = [];

        // Create a more lenient format string from the output date format
        const longFmt = dateFormat.replace('dd', 'd').replace('MM', 'M');
        formats.push(longFmt);

        // Create a version with two-digit year as well
        const shortFmt = longFmt.replace('yyyy', 'yy');
        formats.push(shortFmt);

        // Finally add ISO(ish) formats as last resort
        formats.push("yy-M-d", "yyyy-M-d");

        return formats;
    }

    signal updateRequested(date newValue)

    implicitWidth    : 112
    implicitHeight   : edit.implicitHeight
    activeFocusOnTab : true
    baselineOffset   : edit.baselineOffset

    QtObject {
        id : p_

        property TextInput editor : null

        function isTwoDigitYear(format) {
            return format.indexOf('yyyy') < 0;
        }

        function commitEdit(text) {
            if (allowInvalid && (!text || text.length === 0)) {
                setInvalidDate();
                return;
            }

            var date;
            for (var i = 0; i < parseFormats.length; ++i) {
                date = Date.fromLocaleDateString(
                            locale, text, parseFormats[i]);
                if (date.getFullYear() > 0) {
                    if (isTwoDigitYear(parseFormats[i])) {
                        // Use current century for short date formats
                        var century = new Date().getFullYear();
                        century -= century % 100;
                        date.setFullYear(century + date.getFullYear() % 100);
                    }

                    setDate(date);

                    break;
                }
            }
        }

        function setDate(date) {
            date.setHours(0);
            date.setMinutes(0);
            date.setSeconds(0);
            date.setMilliseconds(0);

            if (uiDateEdit.bindable) {
                uiDateEdit.updateRequested(date);
            } else {
                uiDateEdit.date = date;
            }
        }

        function setInvalidDate() {
            const invalid = new Date(Number.NaN);
            if (uiDateEdit.bindable) {
                uiDateEdit.updateRequested(invalid);
            } else {
                uiDateEdit.date = invalid;
            }
        }

        function formatDate(date) {
            return date.toLocaleDateString(locale, dateFormat);
        }
    }

    UiToolButton {
        id : button

        anchors {
            right : parent.right
            top : parent.top
            bottom : parent.bottom
        }
        width : height - 1
        iconName : "calendar"
        z : 100
        down : popup.popupVisible
        enabled : !uiDateEdit.readOnly

        onClicked : popup.show()
    }

    UiPopupAnchor {
        id : popup

        anchors.left   : parent.left
        anchors.bottom : parent.bottom
        triggerItem    : button
        autoLayout     : true
        onAboutToShow  : uiDateEdit.forceActiveFocus()

        popup : UiCalendar {
            id : calendar

            opacity  : 0
            enabled  : false
            y        : -1
            bindable : true

            date        : uiDateEdit.date
            minimumDate : uiDateEdit.minimumDate
            maximumDate : uiDateEdit.maximumDate

            onUpdateRequested : {
                p_.setDate(newValue);
                popup.hide();
            }

            states : State {
                name : "visible"
                when : popup.popupVisible

                PropertyChanges {
                    target  : calendar
                    opacity : 1
                    enabled : true
                }
            }

            transitions : Transition {
                NumberAnimation {
                    property : "opacity"
                    duration : 100
                }
            }
        }
    }

    UiLineEdit {
        id : edit

        anchors.fill : uiDateEdit
        bindable: true
        focus : true
        rightPadding : button.width + 4
        readOnly : uiDateEdit.readOnly
        text : {
            var now = new Date();
            var today = uiDateEdit.date.getFullYear() === now.getFullYear() &&
                    uiDateEdit.date.getMonth() === now.getMonth() &&
                    uiDateEdit.date.getDate() === now.getDate();

            if (today) {
                return qsTrId("8298533db336ff8c");
            }

            return p_.formatDate(uiDateEdit.date);
        }

        onUpdateRequested : p_.commitEdit(newValue)
    }
}

